-- EpicForge Addon
-- Adds right-click options for copying character, guild links, and character compare links

EpicForge = {}
EpicForge.realm = GetRealmName()

-- Helper to robustly fetch the popup's edit box across game versions
local function GetPopupEditBox(frame)
    -- Retail/older Classic used 'editBox'; newer Classic uses 'EditBox'
    return frame and (frame.editBox or frame.EditBox)
end

-- Helper to robustly fetch the 'data' table (some builds pass it to OnShow, some store on the frame)
local function GetPopupData(self, data)
    if data and type(data) == "table" then
        return data
    end
    if self and type(self.data) == "table" then
        return self.data
    end
    return nil
end

StaticPopupDialogs["EPICFORGE_COPY_LINK"] = {
    text = "Copy & Paste the link into your browser",
    button1 = "Close",
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
    preferredIndex = 3,
    hasEditBox = true,
    OnShow = function(self, data)
        local payload = GetPopupData(self, data)
        local url = (payload and payload.url) or ""

        local editBox = GetPopupEditBox(self)
        if editBox then
            editBox:SetAutoFocus(true)
            editBox:SetText(url)
            editBox:HighlightText()
        else
            -- Fallback: no edit box found (shouldn't happen with hasEditBox=true)
            if self and self.Text and self.Text.SetText then
                self.Text:SetText(url)
            end
        end
    end,
}

local function realmSlugFromName(name)
    -- strip punctuation/control, then spaces -> hyphens, lower
    return (name or GetRealmName()):gsub("[%p%c]", ""):gsub("[%s]", "-"):lower()
end

local function EpicForgeCharacterLinkButton(contextData)
    print("EpicForge: Copy Character Link button pressed")
    local realmSlug = realmSlugFromName(GetRealmName())
    local url = string.format("https://epicforge.au/character/%s/%s", realmSlug, (contextData.name or ""):lower())
    StaticPopup_Show("EPICFORGE_COPY_LINK", "", "", { url = url })
end

local function EpicForgeGuildLinkButton(contextData)
    print("EpicForge: Copy Guild Link button pressed")
    -- Try to find the unit from the menu context; fall back to 'target'
    local unit = (contextData and contextData.unit) or (UIDROPDOWNMENU_INIT_MENU and UIDROPDOWNMENU_INIT_MENU.unit) or "target"
    local guildName = unit and GetGuildInfo(unit) or nil
    if not guildName then
        print("EpicForge: Player is not in a guild.")
        return
    end
    local realmSlug = realmSlugFromName(GetRealmName())
    local formattedGuildName = guildName:lower():gsub(" ", "+")
    local url = string.format("https://epicforge.au/guild/%s/%s", realmSlug, formattedGuildName)
    StaticPopup_Show("EPICFORGE_COPY_LINK", "", "", { url = url })
end

local function EpicForgeCharCompareButton(contextData)
    print("EpicForge: Char Compare button pressed")
    local playerName = (UnitName("player") or ""):lower()
    local playerRealmSlug = realmSlugFromName(GetRealmName())
    local selectedName = (contextData.name or ""):lower()
    -- Assuming same realm for now
    local selectedRealmSlug = playerRealmSlug
    local url = string.format(
        "https://epicforge.au/char_compare.php?char1Name=%s&char1Realm=%s&char2Name=%s&char2Realm=%s",
        playerName, playerRealmSlug, selectedName, selectedRealmSlug
    )
    StaticPopup_Show("EPICFORGE_COPY_LINK", "", "", { url = url })
end

local menuTags = {
    "MENU_UNIT_SELF",
    "MENU_UNIT_PARTY",
    "MENU_UNIT_PLAYER",
    "MENU_UNIT_ENEMY_PLAYER",
    "MENU_UNIT_RAID_PLAYER",
    "MENU_UNIT_RAID",
    "MENU_UNIT_FRIEND",
    "MENU_UNIT_FRIEND_OFFLINE",
    "MENU_UNIT_BN_FRIEND",
    "MENU_UNIT_GUILD",
    "MENU_UNIT_GUILD_OFFLINE",
    "MENU_UNIT_CHAT_ROSTER",
    "MENU_UNIT_ARENAENEMY",
    "MENU_UNIT_FOCUS",
    "MENU_UNIT_COMMUNITIES_WOW_MEMBER",
    "MENU_UNIT_COMMUNITIES_GUILD_MEMBER",
    "MENU_UNIT_GUILDS_GUILD",
    "MENU_UNIT_COMMUNITIES_MEMBER",
    "MENU_UNIT_COMMUNITIES_COMMUNITY",
    "MENU_UNIT_RAID_TARGET_ICON",
    "MENU_UNIT_WORLD_STATE_SCORE",
    "MENU_UNIT_PVP_SCOREBOARD",
}

for _, tag in ipairs(menuTags) do
    -- Menu.ModifyMenu provided by Blizzard's new Menu system
    Menu.ModifyMenu(tag, function(ownerRegion, rootDescription, contextData)
        rootDescription:CreateDivider()
        rootDescription:CreateButton("|cfff8b700Copy Character Link", function()
            EpicForgeCharacterLinkButton(contextData)
        end)
        rootDescription:CreateButton("|cfff8b700Copy Guild Link", function()
            EpicForgeGuildLinkButton(contextData)
        end)
        rootDescription:CreateButton("|cfff8b700Char Compare", function()
            EpicForgeCharCompareButton(contextData)
        end)
    end)
end
